import os
import ssl
import smtplib
import logging
import requests
import socket
import time
import imaplib
import email
from email.header import decode_header, make_header
from email.utils import parseaddr, formatdate
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from typing import Optional, List, Dict, Any
from livekit.agents import function_tool, RunContext

LOG = logging.getLogger(__name__)

# ---------- helpers ----------
def _env(name: str, default: Optional[str] = None) -> str:
    v = os.getenv(name, default)
    if v is None:
        raise RuntimeError(f"Missing required environment variable: {name}")
    return v

def _decode(s):
    try:
        return str(make_header(decode_header(s))) if s else ""
    except Exception:
        return s or ""

def _imap_login() -> imaplib.IMAP4_SSL:
    user = _env("GMAIL_ADDRESS")
    app_pw = _env("GMAIL_APP_PASSWORD")
    ctx = ssl.create_default_context()
    M = imaplib.IMAP4_SSL("imap.gmail.com", 993, ssl_context=ctx)
    rv, _ = M.login(user, app_pw)
    if rv != "OK":
        raise RuntimeError("IMAP login failed")
    return M

# ---------- weather ----------
@function_tool()
async def get_weather(context: RunContext, city: str) -> str:
    try:
        geo = requests.get("https://geocoding-api.open-meteo.com/v1/search",
                           params={"name": city, "count": 1}, timeout=10).json()
        if not geo.get("results"):
            return f"Sorry, I couldn't find {city}."
        lat = geo["results"][0]["latitude"]
        lon = geo["results"][0]["longitude"]
        wx = requests.get("https://api.open-meteo.com/v1/forecast",
                          params={"latitude": lat, "longitude": lon, "current_weather": True},
                          timeout=10).json()
        cw = wx.get("current_weather") or {}
        if not cw:
            return f"Weather unavailable for {city}."
        return f"{city}: {cw.get('temperature')}°C, wind {cw.get('windspeed')} km/h."
    except Exception as e:
        LOG.exception("weather error")
        return f"Weather lookup failed: {e}"

# ---------- email send (robust SMTP) ----------
def _smtp_send(msg: MIMEMultipart, recipients: List[str], gmail_user: str, gmail_app_pw: str, use_ipv4: bool=False) -> str:
    ctx = ssl.create_default_context()
    orig_getaddrinfo = socket.getaddrinfo
    if use_ipv4:
        def only_ipv4_getaddrinfo(*args, **kwargs):
            return [i for i in orig_getaddrinfo(*args, **kwargs) if i[0] == socket.AF_INET]
        socket.getaddrinfo = only_ipv4_getaddrinfo  # type: ignore
    try:
        try:
            with smtplib.SMTP_SSL("smtp.gmail.com", 465, timeout=30, context=ctx) as s:
                s.login(gmail_user, gmail_app_pw)
                s.sendmail(gmail_user, recipients, msg.as_string())
                return "465/SSL"
        except Exception as e:
            LOG.warning("465 SSL path failed: %s", e)
        with smtplib.SMTP("smtp.gmail.com", 587, timeout=30) as s:
            s.ehlo()
            s.starttls(context=ctx)
            s.login(gmail_user, gmail_app_pw)
            s.sendmail(gmail_user, recipients, msg.as_string())
            return "587/STARTTLS"
    finally:
        if use_ipv4:
            socket.getaddrinfo = orig_getaddrinfo

@function_tool()
async def send_email(context: RunContext, to_email: str, subject: str, message: str, cc_email: Optional[str] = None) -> str:
    gmail_user = _env("GMAIL_ADDRESS")
    gmail_app_pw = _env("GMAIL_APP_PASSWORD")
    msg = MIMEMultipart()
    msg["From"] = gmail_user
    msg["To"] = to_email
    if cc_email:
        msg["Cc"] = cc_email
    msg["Subject"] = subject
    msg["Date"] = formatdate(localtime=True)
    msg.attach(MIMEText(message, "plain"))
    recipients = [r for r in [to_email, cc_email] if r]
    delays = [0.5, 1.5, 3.0, 6.0]
    last_err: Optional[Exception] = None
    for i, d in enumerate(delays, 1):
        try:
            path = _smtp_send(msg, recipients, gmail_user, gmail_app_pw, use_ipv4=(i == len(delays)))
            return f"Email sent via {path} to {', '.join(recipients)}."
        except smtplib.SMTPAuthenticationError:
            return "Authentication failed — check GMAIL_ADDRESS and GMAIL_APP_PASSWORD."
        except (smtplib.SMTPConnectError, smtplib.SMTPServerDisconnected, smtplib.SMTPException, TimeoutError, OSError) as e:
            last_err = e; LOG.warning("SMTP error (attempt %d/%d): %s; retrying in %.1fs", i, len(delays), e, d); time.sleep(d)
        except Exception as e:
            last_err = e; LOG.warning("Unexpected send error (attempt %d/%d): %s; retrying in %.1fs", i, len(delays), e, d); time.sleep(d)
    LOG.exception("send_email error")
    return f"Failed to send after retries: {last_err}"

@function_tool()
async def gmail_smtp_healthcheck(context: RunContext) -> str:
    gmail_user = _env("GMAIL_ADDRESS")
    gmail_app_pw = _env("GMAIL_APP_PASSWORD")
    ctx = ssl.create_default_context()
    results = []
    try:
        with smtplib.SMTP_SSL("smtp.gmail.com", 465, timeout=15, context=ctx) as s:
            s.login(gmail_user, gmail_app_pw)
        results.append("465/SSL: OK")
    except Exception as e:
        results.append(f"465/SSL: {e.__class__.__name__}: {e}")
    try:
        with smtplib.SMTP("smtp.gmail.com", 587, timeout=15) as s:
            s.ehlo(); s.starttls(context=ctx); s.login(gmail_user, gmail_app_pw)
        results.append("587/STARTTLS: OK")
    except Exception as e:
        results.append(f"587/STARTTLS: {e.__class__.__name__}: {e}")
    return " | ".join(results)

# ---------- Gmail IMAP: search/list/read/reply ----------
@function_tool()
async def gmail_search(context: RunContext, query: Optional[str]=None, since: Optional[str]=None, before: Optional[str]=None, max_results: int=10) -> List[Dict[str, Any]]:
    M = _imap_login()
    try:
        M.select("INBOX")
        parts = []
        if query:
            parts += ['OR', 'SUBJECT', f'"{query}"', 'BODY', f'"{query}"']
        if since: parts += ["SINCE", since]
        if before: parts += ["BEFORE", before]
        if not parts: parts = ["ALL"]
        rv, data = M.search(None, *parts)
        if rv != "OK": return []
        uids = list(reversed(data[0].split()))[:max_results]
        results: List[Dict[str, Any]] = []
        for uid in uids:
            rv, msgdata = M.fetch(uid, "(RFC822)")
            if rv != "OK": continue
            msg = email.message_from_bytes(msgdata[0][1])
            results.append({
                "uid": uid.decode(),
                "from": parseaddr(msg.get("From"))[1],
                "subject": _decode(msg.get("Subject")),
                "date": msg.get("Date") or "",
                "message_id": msg.get("Message-ID") or "",
            })
        return results
    finally:
        try: M.close()
        except Exception: pass
        M.logout()

@function_tool()
async def gmail_read_by_uid(context: RunContext, uid: str) -> Dict[str, Any]:
    M = _imap_login()
    try:
        M.select("INBOX")
        rv, msgdata = M.fetch(uid, "(RFC822)")
        if rv != "OK": return {"error": "fetch failed"}
        msg = email.message_from_bytes(msgdata[0][1])
        body = ""
        if msg.is_multipart():
            for part in msg.walk():
                ctype = part.get_content_type()
                disp = str(part.get("Content-Disposition") or "")
                if ctype == "text/plain" and "attachment" not in disp:
                    body = part.get_payload(decode=True).decode(part.get_content_charset() or "utf-8", errors="replace"); break
        else:
            if msg.get_content_type() == "text/plain":
                body = msg.get_payload(decode=True).decode(msg.get_content_charset() or "utf-8", errors="replace")
        return {
            "uid": uid,
            "from": parseaddr(msg.get("From"))[1],
            "subject": _decode(msg.get("Subject")),
            "date": msg.get("Date") or "",
            "message_id": msg.get("Message-ID") or "",
            "body": (body or "").strip(),
        }
    finally:
        try: M.close()
        except Exception: pass
        M.logout()

@function_tool()
async def gmail_reply_by_uid(context: RunContext, uid: str, reply_text: str, cc_email: Optional[str]=None) -> str:
    info = await gmail_read_by_uid(context, uid)
    if "error" in info: return f"Cannot fetch original: {info['error']}"
    gmail_user = _env("GMAIL_ADDRESS"); gmail_app_pw = _env("GMAIL_APP_PASSWORD")
    msg = MIMEMultipart()
    msg["From"] = gmail_user; msg["To"] = info["from"]
    if cc_email: msg["Cc"] = cc_email
    msg["Subject"] = f"Re: {info['subject']}"
    if info.get("message_id"):
        msg["In-Reply-To"] = info["message_id"]; msg["References"] = info["message_id"]
    msg["Date"] = formatdate(localtime=True); msg.attach(MIMEText(reply_text, "plain"))
    recipients = [r for r in [info["from"], cc_email] if r]
    try:
        path = _smtp_send(msg, recipients, gmail_user, gmail_app_pw)
        return f"Reply sent via {path} to {', '.join(recipients)}."
    except smtplib.SMTPAuthenticationError:
        return "Authentication failed — check GMAIL_ADDRESS and GMAIL_APP_PASSWORD."
    except Exception as e:
        LOG.exception("gmail_reply_by_uid error"); return f"Failed to send reply: {e}"

# ---------- n8n ----------
@function_tool()
async def trigger_n8n(context: RunContext, webhook_url: str, payload_json: str) -> str:
    try:
        r = requests.post(webhook_url, data=payload_json, headers={"Content-Type": "application/json"}, timeout=30)
        return f"n8n responded {r.status_code}: {r.text[:200]}..."
    except Exception as e:
        LOG.exception("n8n trigger error")
        return f"n8n trigger failed: {e}"